# Static Sites on Fly.io

## Using nginx

Best for pure HTML/CSS/JS sites.

### Dockerfile

```dockerfile
FROM nginx:alpine

# Copy static files
COPY . /usr/share/nginx/html

# Copy nginx config
COPY nginx.conf /etc/nginx/nginx.conf

EXPOSE 8080
```

### nginx.conf

```nginx
events {
    worker_connections 1024;
}

http {
    include /etc/nginx/mime.types;
    default_type application/octet-stream;
    
    server {
        listen 8080;
        server_name _;
        root /usr/share/nginx/html;
        index index.html;
        
        # SPA fallback
        location / {
            try_files $uri $uri/ /index.html;
        }
        
        # Cache static assets
        location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2)$ {
            expires 1y;
            add_header Cache-Control "public, immutable";
        }
        
        # Health check
        location /health {
            access_log off;
            return 200 'OK';
            add_header Content-Type text/plain;
        }
    }
}
```

### fly.toml

```toml
app = "my-static-site"
primary_region = "ord"

[build]
  dockerfile = "Dockerfile"

[http_service]
  internal_port = 8080
  force_https = true
  auto_stop_machines = "stop"
  auto_start_machines = true

[[http_service.checks]]
  grace_period = "5s"
  interval = "30s"
  method = "GET"
  path = "/health"
  timeout = "5s"
```

## SPA Frameworks (React, Vue, etc.)

For built SPA output (dist/build folder):

### Dockerfile

```dockerfile
FROM node:20-alpine AS builder
WORKDIR /app
COPY package*.json ./
RUN npm ci
COPY . .
RUN npm run build

FROM nginx:alpine
COPY --from=builder /app/dist /usr/share/nginx/html
COPY nginx.conf /etc/nginx/nginx.conf
EXPOSE 8080
```

## Alternative: Using Caddy

Simpler config, automatic HTTPS:

### Dockerfile

```dockerfile
FROM caddy:alpine

COPY Caddyfile /etc/caddy/Caddyfile
COPY . /srv

EXPOSE 8080
```

### Caddyfile

```
:8080 {
    root * /srv
    file_server
    
    # SPA fallback
    try_files {path} /index.html
    
    # Health check
    respond /health 200
}
```

## Minimal: Single HTML File

For very simple sites:

### Dockerfile

```dockerfile
FROM nginx:alpine
COPY index.html /usr/share/nginx/html/
EXPOSE 80
```

### fly.toml

```toml
app = "my-simple-site"
primary_region = "ord"

[build]
  dockerfile = "Dockerfile"

[http_service]
  internal_port = 80
  force_https = true
  auto_stop_machines = "stop"
  auto_start_machines = true
```

## Common Issues

### 404 on Refresh (SPA)

**Cause:** Server returns 404 for client-side routes.

**Fix:** Add SPA fallback in nginx.conf:
```nginx
location / {
    try_files $uri $uri/ /index.html;
}
```

### Wrong MIME Types

**Symptom:** CSS/JS not loading, browser shows wrong content type.

**Fix:** Ensure nginx includes MIME types:
```nginx
http {
    include /etc/nginx/mime.types;
    default_type application/octet-stream;
    ...
}
```

### Large Files

For sites with large assets, increase client_max_body_size:
```nginx
http {
    client_max_body_size 100M;
    ...
}
```
