# Docker Deployment on Fly.io

## Basic Deployment

If the project has a working Dockerfile:

```bash
fly launch --no-deploy
# Review fly.toml, then:
fly deploy
```

## Dockerfile Best Practices

### Multi-Stage Builds

Reduce image size with multi-stage builds:

```dockerfile
# Build stage
FROM node:20-alpine AS builder
WORKDIR /app
COPY package*.json ./
RUN npm ci
COPY . .
RUN npm run build

# Production stage
FROM node:20-alpine
WORKDIR /app
ENV NODE_ENV=production
COPY --from=builder /app/dist ./dist
COPY --from=builder /app/node_modules ./node_modules
EXPOSE 8080
CMD ["node", "dist/index.js"]
```

### Non-Root User

Run as non-root for security:

```dockerfile
RUN addgroup --system --gid 1001 appgroup
RUN adduser --system --uid 1001 appuser
USER appuser
```

### Listen on 0.0.0.0

Apps must bind to all interfaces:

```dockerfile
# Ensure app listens on 0.0.0.0, not 127.0.0.1
ENV HOST=0.0.0.0
ENV PORT=8080
EXPOSE 8080
```

## fly.toml Configuration

```toml
app = "my-docker-app"
primary_region = "ord"

[build]
  dockerfile = "Dockerfile"
  # Or custom path:
  # dockerfile = "deploy/Dockerfile"

[http_service]
  internal_port = 8080  # Must match EXPOSE in Dockerfile
  force_https = true
  auto_stop_machines = "stop"
  auto_start_machines = true

[[http_service.checks]]
  grace_period = "15s"
  interval = "30s"
  method = "GET"
  path = "/health"
  timeout = "5s"
```

## Build Arguments

Pass build-time variables:

```toml
[build.args]
  NODE_ENV = "production"
  BUILD_VERSION = "1.0.0"
```

Use in Dockerfile:
```dockerfile
ARG NODE_ENV=development
ARG BUILD_VERSION
ENV NODE_ENV=$NODE_ENV
```

## .dockerignore

Always include to speed up builds:

```
.git
node_modules
*.md
.env*
.DS_Store
coverage
.next
dist
```

## Pre-Built Images

Deploy an existing image without building:

```toml
[build]
  image = "nginx:alpine"
```

Or from a registry:
```toml
[build]
  image = "ghcr.io/myorg/myapp:latest"
```

## Common Issues

### Port Mismatch

**Symptom:** Health checks fail, app seems to run.

**Fix:** Ensure `internal_port` matches `EXPOSE` and actual app port.

### Build Context Too Large

**Symptom:** Build takes forever to upload.

**Fix:** Add `.dockerignore` to exclude large/unnecessary files.

### Can't Find Entrypoint

**Symptom:** Container exits immediately.

**Fix:** Ensure Dockerfile has CMD or ENTRYPOINT:
```dockerfile
CMD ["node", "server.js"]
# Or
ENTRYPOINT ["./start.sh"]
```

### Secrets Not Available at Build Time

Build-time secrets use `--build-secret`:
```bash
fly deploy --build-secret MY_TOKEN=xxx
```

In Dockerfile:
```dockerfile
RUN --mount=type=secret,id=MY_TOKEN \
    cat /run/secrets/MY_TOKEN
```

Runtime secrets use `fly secrets set` (available as env vars).
